define(['angular', 'app', 'modalService'], function (angular, modalService) {
	'use strict';

	// modified from: https://github.com/facultymatt/angular-unsavedChanges
	angular.module('unsavedChanges', [])

		.provider('unsavedWarningsConfig', function () {

			var _this = this;
			var useTranslateService = true;
			var routeEvent = ['$locationChangeStart', '$stateChangeStart'];
			var navigateMessage = 'You have not saved your changes. If you wish to continue without saving select \'CONTINUE WITHOUT SAVING\', otherwise select \'RETURN TO FORM\' to return to the entry screen.';
			var reloadMessage = "You have not saved your changes. You may continue without saving or return to the entry screen.";
			var modalOptions = {
				closeButtonText: 'Return to Form',
				actionButtonText: 'Continue without Saving',
				headerText: 'Confirmation',
				bodyText: navigateMessage
			};

			Object.defineProperty(_this, 'navigateMessage', {
				get: function () {
					return navigateMessage;
				},
				set: function (value) {
					navigateMessage = value;
				}
			});

			Object.defineProperty(_this, 'reloadMessage', {
				get: function () {
					return reloadMessage;
				},
				set: function (value) {
					reloadMessage = value;
				}
			});

			Object.defineProperty(_this, 'useTranslateService', {
				get: function () {
					return useTranslateService;
				},
				set: function (value) {
					useTranslateService = !!(value);
				}
			});

			Object.defineProperty(_this, 'routeEvent', {
				get: function () {
					return routeEvent;
				},
				set: function (value) {
					if (typeof value === 'string') value = [value];
					routeEvent = value;
				}
			});

			Object.defineProperty(_this, 'modalOptions', {
				get: function () {
					return modalOptions;
				},
				set: function (value) {
					modalOptions = value;
				}
			});

			this.$get = ['$injector',
				function ($injector) {

					function translateIfAble(message) {
						if ($injector.has('$translate') && useTranslateService) {
							return $injector.get('$translate')(message);
						} else {
							return false;
						}
					}

					var publicInterface = {};

					Object.defineProperty(publicInterface, 'useTranslateService', {
						get: function () {
							return useTranslateService;
						}
					});

					Object.defineProperty(publicInterface, 'reloadMessage', {
						get: function () {
							return translateIfAble(reloadMessage) || reloadMessage;
						}
					});

					Object.defineProperty(publicInterface, 'navigateMessage', {
						get: function () {
							return translateIfAble(navigateMessage) || navigateMessage;
						}
					});

					Object.defineProperty(publicInterface, 'routeEvent', {
						get: function () {
							return routeEvent;
						}
					});

					Object.defineProperty(publicInterface, 'modalOptions', {
						get: function () {
							return modalOptions;
						}
					});

					return publicInterface;
				}
			];
		})

		.service('unsavedWarningSharedService',
		function ($rootScope, $state, unsavedWarningsConfig, modalService, focusService) {

			var _this = this;
			var allForms = [];
			var areAllFormsClean = true;
			var removeFunctions = [angular.noop];
			var messages = {
				navigate: unsavedWarningsConfig.navigateMessage,
				reload: unsavedWarningsConfig.reloadMessage
			};

			this.allFormsClean = function () {
				areAllFormsClean = true;
				angular.forEach(allForms, function (item) {

					if (item.$dirty) {
						areAllFormsClean = false;
					}
				});
				return areAllFormsClean; // no dirty forms were found
			};

			this.cleanAllForms = function () {
				angular.forEach(allForms, function (item) {
					if (item.$dirty) {
						item.$setPristine();
					}
				});
			};

			this.getDirtyForms = function () {
				return allForms.filter(function (item) {
					return item.$dirty;
				});
			};

			this.getCleanForms = function () {
				return allForms.filter(function (item) {
					return !item.$dirty;
				});
			};

			// adds form controller to registered forms array
			// this array will be checked when user navigates away from page
			this.init = function (form) {
				if (allForms.length === 0) setup();
				allForms.push(form);
			};

			this.removeForm = function (form) {
				var idx = allForms.indexOf(form);

				// this form is not present array
				if (idx === -1) return;

				allForms.splice(idx, 1);

				if (allForms.length === 0) tearDown();
			};

			function tearDown() {
				angular.forEach(removeFunctions, function (fn) {
					fn();
				});
				window.onbeforeunload = null;
			}

			// Function called when user tries to close the window
			this.confirmExit = function () {
				if (!_this.allFormsClean()) return messages.reload;
				tearDown();
			};

			function setup() {
				window.onbeforeunload = _this.confirmExit;

				var eventsToWatchFor = unsavedWarningsConfig.routeEvent;

				angular.forEach(eventsToWatchFor, function (aEvent) {
					//calling this function later will unbind this, acting as $off()

					var removeFn = $rootScope.$on(aEvent, function (event, toState, toParams) {

						if (!_this.allFormsClean()) {
							event.preventDefault();

							modalService.showModal({}, unsavedWarningsConfig.modalOptions).then(function () {

								_this.cleanAllForms();

								if (event.name === "$stateChangeStart") {

									$state.go(toState.name, toParams);
								}
							}, function(){
								focusService.focusPrimary();
							});
						}
					});
					removeFunctions.push(removeFn);
				});
			}
		}
	)

		.directive('unsavedWarningClear',
		function () {
			return {
				scope: {},
				require: '^form',
				priority: 10,
				link: function (scope, element, attrs, formCtrl) {
					element.bind('click', function () {
						formCtrl.$setPristine();
					});
				}
			};
		}
	)

		.directive('unsavedWarningTrigger',
		function (unsavedWarningSharedService, $rootScope, unsavedWarningsConfig, modalService, focusService) {
			return {
				restrict: 'A',
				priority: 1,
				link: function (scope, element, attr) {
					var clickAction = attr.ngClick;
					element.unbind('click');
					element.bind('click', function () {
						if (!unsavedWarningSharedService.allFormsClean()) {
							modalService.showModal({}, unsavedWarningsConfig.modalOptions).then(function () {
								unsavedWarningSharedService.cleanAllForms();
								scope.$eval(clickAction);
							}, function(){
								focusService.focusPrimary();
							});
						} else {
							scope.$eval(clickAction);
						}
					});
				}
			};
		}
	)

		.directive('unsavedWarningForm',
		function (unsavedWarningSharedService) {
			return {
				scope: {},
				require: 'form',
				link: function (scope, formElement, attrs, formCtrl) {

					// register this form
					unsavedWarningSharedService.init(formCtrl);

					// bind to form submit, this makes the typical submit button work
					// in addition to the ability to bind to a seperate button which clears warning
					formElement.bind('submit', function () {
						if (formCtrl.$valid) {
							formCtrl.$setPristine();
						}
					});

					formElement.bind('reset', function (event) {
						event.preventDefault();

						// sets for back to valid and pristine states
						formCtrl.$setPristine();
					});

					// @todo check destroy on clear button too?
					scope.$on('$destroy', function () {
						unsavedWarningSharedService.removeForm(formCtrl);
					});
				}
			};
		}
	);
});